%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This work by EPFL STI IBI LBNI is licensed under 
% a Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.
% Based on a work at http://lbni.epfl.ch/.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


function [ channel_info ] = getChannelInfo(Filename, ImageDir, BaseDirectory)
%getChannelInfo finds the different channels in the input file (given by
%Filname of the directory ImageDir), and returns the information concerning
%the channels

    warning('off','MATLAB:MKDIR:DirectoryExists');
    file_name = [ImageDir,Filename{1}];
    mkdir( [ImageDir,BaseDirectory,filesep]);
    
    searchstring(1).label='@2:Z scale:';
    searchstring(2).label='Samps';
    searchstring(3).label='Lines:';
    searchstring(4).label='Data offset';
    searchstring(5).label='Image Data:';
    searchstring(6).label='Scan Size:';
    searchstring(7).label='@Sens. Zsens:';
    searchstring(8).label='@Z magnify:';
    
    searchstring(9).label='Scan Rate:';
    searchstring(10).label='@Sens. DeflSens: V';
    searchstring(11).label='@2:CantFrequency: V';
    searchstring(12).label='@2:CantDrive: V';
    searchstring(13).label='@2:TMSetAmplitude: V';
    searchstring(14).label='@Sens. ZsensSens: V';
    searchstring(15).label='@Sens. Amplitude Error: V';
    searchstring(16).label='@Sens. Amplitude: V';
    searchstring(17).label='@Sens. ForceDeflSens: V';
    searchstring(18).label='Aspect Ratio:';

%%% Read in DI Image and image propreties  This code is mostly from Jaco De Groot at University College London

    % openNano version 6.0
    % Last updated: 01-06-2006 by Macerano Blanco for correctly reading
    % Nanoscope 6 files
    % 
    % [images Channel_Info ]=openNano(filename)
    %
    % Reads images from a Nanoscope 6 image files. The images are stored in
    % "images" and can be displayed with imshow(images(:,:,channel_number). The
    % relevant channel number can be found in the print output fromt this subroutine
    % Basic channel information can be found in Channel_Info that for each
    % channel contains, the name, trace/retrace, unit and x and y scan size
    %
    % Example:
    % [images channel_info]=openNano('C:\Briefcase\Parchment\AFM\Dimension\CR41\CR41.000');
    % The output of this command is:
    %  Channel                                     Unit    
    %   1. Height                   Trace          um
    %   2. Deflection               Trace          V
    %
    % The following command images the trace height. The height is specified in
    % microns:
    % imshow(images(:,:,1),[]);
    % This image can be stored as a gray scale image with
    % imwrite(mat2gray(images(:,:,1)),'g:/traceheight.tif','tif');

    % This function/script is authorized for use in government and academic
    % research laboratories and non-profit institutions only. Though this
    % function has been tested prior to its posting, it may contain mistakes or
    % require improvements. In exchange for use of this free product, we 
    % request that its use and any issues that may arise be reported to us. 
    % Comments and suggestions are therefore welcome and should be sent to 
    % 
    % Jaco de Groot
    % Bone & Mineral Centre, Dept of Medicine
    % University College London
    % 5 University Street
    % The Rayne Building
    % London    WC1E 6JJ
    % Tel: 020-7679 6143
    % Mob: 07745-948245  
    % Fax: 020-7679 6219
    % Email: jacodegroot@yahoo.se
    % http://www.ucl.ac.uk/medicine/bmc/


    % Define End of file identifier
        % Opend the file given in argument and reference as
        % fid.  Also if there was an error output error
        % number and error message to screen


    fid = fopen(file_name,'r');
        [message,errnum] = ferror(fid);
        if(errnum)
            fprintf(1,'I/O Error %d \t %s',[errnum,message]);
            %   break
        end
        header_end=0; eof = 0; counter = 1; byte_location = 0;
        nstrings=size(searchstring,2);
        parcounter = ones(nstrings,1);
        paramters = struct('trace',cell(1,nstrings),'values',cell(1,nstrings),'channel',cell(1,nstrings));

        for ij=1:nstrings 
            parcounter(ij)=1; 
            parameters(ij).trace=0;
        end;

        while( and( ~eof, ~header_end ) )

        byte_location = ftell(fid);  
        line = fgets(fid);

        for ij=1:nstrings
            if strfind(lower(line),lower(searchstring(ij).label))
                if (extract_num(line))
                    b = strfind(line, 'LSB');
                    if (b>0)
                        parameters(ij).values(parcounter(ij))=extract_num(line(b(1):end));
                    else
                        parameters(ij).values(parcounter(ij))=extract_num(line);
                    end;

                
                elseif strcmp(searchstring(ij).label,'Aspect Ratio:')
                    b= strfind(line,':');
                    parameters(ij).values(parcounter(ij))=extract_num(line(b(1):b(2)-1));

               
                else
                    b= strfind(lower(line),'"');
                    if (b>0)  
                        parameters(ij).channel(parcounter(ij)).name=line(b(1)+1:b(2)-1); 
                    else
                        if (strfind(line,'Trace')>0) 
                            parameters(ij).trace=1; 
                        end;
                    end;
                end;
                parcounter(ij)=parcounter(ij)+1;
            end
        end;

        if( (-1)==line )  
            eof  = 1;  
        end

        if ~isempty( strfind( line, '\*File list end' ) ) 
            header_end = 1;   
        end

        counter=counter+1;
        end
        
    fclose(fid);

    param           = parameters;
    scaling         = param(1).values;                                                                                                                                                 % Scaling parameters
    spl             = param(2).values;                                                                                                                                                 % Samples per line
    linno           = param(3).values;                                                                                                                                                 % No of lines 
    image_pos       = param(4).values;                                                                                                                                               % Data position
    ScanSize        = param(6).values(1);
    Z_Sensitivity   = param(7).values;                                                                                                                                                 % Data position
    Z_Magnification = param(8).values;                                                                                                                                                 % Zmagnification
    
    Scan_Rate       = param(9).values(1);
    Defl_Sens       = param(10).values;
    Drive_Freq      = param(11).values;
    Drive_Amp       = param(12).values;
    Amp_Set_Point   = param(13).values;
    
    [tok remain]    = strtok(Filename{1},'.');
    file_number     = str2double(remain(2:4));
    
    Image_Data_File = [ImageDir,BaseDirectory,filesep,'Imaging_Parameters.txt']; 
    fid = fopen(Image_Data_File, 'a');
        fprintf(fid, [  'File Number','\t',...
                        'Samples per Line','\t',...
                        'Lines per Image','\t',...
                        'Scan Rate (Hz)','\t',...
                        'Deflection Sensitivity (nm/V)','\t',...
                        'Drive Frequency (kHz)','\t',...
                        'Drive Amplitude (mV)','\t',...
                        'Amplitude Setpoint (mV)','\n'],...
                        'char');
                    
        fprintf(fid, [  num2str(file_number,'%0.0f'),'\t',...
                        num2str(spl(1),'%0.0f'),'\t',...
                        num2str(linno,'%0.0f'),'\t',...
                        num2str(Scan_Rate,'%0.3f'),'\t',...
                        num2str(Defl_Sens,'%0.3f'),'\t',...
                        num2str(Drive_Freq,'%0.2f'),'\t',...
                        num2str(Drive_Amp,'%0.2f'),'\t',...
                        num2str(Amp_Set_Point,'%0.2f'),'\n'],...
                        'char');
        fclose(fid);
    
    L = length(image_pos);                                                                                                                                                             % Sets the number of image channels
    channel_info = struct('Trace',cell(L,1),'Name',cell(L,1),'Finalscaling',cell(L,1),'Unit',cell(L,1));
    finalscaling = zeros(L,1);
    
    for im=1:L                                                                                                                                                                         % Loops for the number of channels
        channel_info(im).Trace=char((param(6).trace)*'Trace  '+(1-param(6).trace)*'Retrace');                                                                                          % Determines if Channel is Trace or Retrace
        channel_info(im).Name=param(5).channel(im).name;                                                                                                                               % Name of channel 1
        channel_info(im).Finalscaling(im)=param(8).values(im)*param(1).values(im);                                                                                                     % Computes some of the scaling paramters

        switch channel_info(im).Name                                                                                                                                                   % Sets Units according to recorded channel
            case 'Height'                                                                                                                                                              % If height...
               channel_info(im).Unit='nm' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity)/(2^16);                                                                                                                    % Sets bit scaling

            case 'Deflection'                                                                                                                                                          % If Deflection...
               channel_info(im).Unit='V';                                                                                                                                              % Units are V
               finalscaling(im)=(scaling(im))/(2^16);                                                                                                                                  % Sets bit scaling                                                  
            
            case 'Phase'                                                                                                                                                               % If Phase...
                channel_info(im).Unit='Degree';                                                                                                                                        % Units are degrees
                finalscaling(im)=(scaling(im))/(2^16);                                                                                                                                 % Sets bit scaling
            
            case 'Amplitude'                                                                                                                                                           % If Potential...
                channel_info(im).Unit='mV';                                                                                                                                             % Units are V
                channel_info(im).Finalscaling(im)=param(8).values(im)*param(1).values(im);                                                                                             % Sets final scaling value
                finalscaling(im)=(scaling(im))/(2^16); 
                
            case 'Amplitude Error'                                                                                                                                                           % If Potential...
                channel_info(im).Unit='mV';                                                                                                                                             % Units are V
                channel_info(im).Finalscaling(im)=param(8).values(im)*param(1).values(im);                                                                                             % Sets final scaling value
                finalscaling(im)=(scaling(im))/(2^16); 
                
            case 'Potential'                                                                                                                                                           % If Potential...
                channel_info(im).Unit='V';                                                                                                                                             % Units are V
                channel_info(im).Finalscaling(im)=param(8).values(im)*param(1).values(im);                                                                                             % Sets final scaling value
                finalscaling(im)=(scaling(im))/(2^16);                                                                                                                                 % Sets bit scaling
            
            case 'Peak Force Error'                                                                                                                                                              % If height...
               channel_info(im).Unit='nN' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity)/(2^16); 
            
            case 'DMTModulus'                                                                                                                                                              % If height...
               channel_info(im).Unit='MPa' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity)/(2^16); 
            
            case 'LogDMTModulus'                                                                                                                                                              % If height...
               channel_info(im).Unit='log(MPa)' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity)/(2^16); 
           
            case 'Adhesion'                                                                                                                                                              % If height...
               channel_info(im).Unit='nN  ' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity)/(2^16); 
            
            case 'Deformation'                                                                                                                                                              % If height...
               channel_info(im).Unit='nm' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity)/(2^16); 
            
            case 'Dissipation'                                                                                                                                                              % If height...
               channel_info(im).Unit='eV' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity)/(2^16); 
            
            case 'Input1'                                                                                                                                                              % If height...
               channel_info(im).Unit='nm' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity*22*Defl_Sens/2)/(2^16); 
            

            case 'Input2'                                                                                                                                                              % If height...
               channel_info(im).Unit='nm' ;                                                                                                                                            % Units are nm
               channel_info(im).Finalscaling(im)=param(7).values*param(8).values(im)*param(1).values(im);                                                                              % Sets final scaling value
               finalscaling(im)=(scaling(im)*Z_Sensitivity*22*Defl_Sens)/(2^16); 
               
            otherwise                                                                                                                                                                  % If unknown/other
                channel_info(im).Unit='V';                                                                                                                                             % Units are V
                channel_info(im).Finalscaling(im)=param(8).values(im)*param(1).values(im);                                                                                             % Sets final scaling value
                finalscaling(im)=(scaling(im))/(2^16);                                                                                                                                 % Sets bit scaling
        end         
    end   

assignin('base', 'channel_info_stock', channel_info);    
end

